<?php
declare(strict_types=1);

// Siempre responder JSON
header('Content-Type: application/json; charset=utf-8');

// Evita que warnings/notices se impriman y rompan el JSON
ini_set('display_errors', '0');
error_reporting(E_ALL);

// Helper de salida
function json_exit(array $payload, int $code = 200): void {
    http_response_code($code);
    echo json_encode($payload, JSON_UNESCAPED_UNICODE);
    exit();
}

function verificarCodigoDescuento(string $codigo): bool {
    $archivo = __DIR__ . '/codigos_descuento.txt';
    if (!file_exists($archivo)) return false;
    $codigosValidos = file($archivo, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES) ?: [];
    return in_array($codigo, $codigosValidos, true);
}

function escribirCSV(array $datos): bool {
    $archivo = __DIR__ . '/registros.csv';
    $manejador = @fopen($archivo, 'a');
    if (!$manejador) return false;
    $ok = fputcsv($manejador, $datos) !== false;
    fclose($manejador);
    return $ok;
}

function escribirTXT(string $contenido): bool {
    $archivo = __DIR__ . '/registros.txt';
    return @file_put_contents($archivo, $contenido, FILE_APPEND) !== false;
}

function enviarCorreo(string $name, string $lastname, string $rut, string $phone, string $email, string $birthdate, string $address, string $region, string $subscription, string $discount_code): bool {
    $correo_remitente = "jano@mauroquintana.cl";
    $nombre_remitente = "CEM Mauro Quintana";
    $destinatario = "jano@mauroquintana.cl";
    $asunto = "Venta Precalculo $subscription";

    $mensaje = "Se ha recibido un nuevo pedido:\n\n";
    $mensaje .= "Nombre(s): $name\n";
    $mensaje .= "Apellidos: $lastname\n";
    $mensaje .= "RUT: $rut\n";
    $mensaje .= "Teléfono: +56 9 $phone\n";
    $mensaje .= "Correo Electrónico: $email\n";
    $mensaje .= "Fecha de nacimiento: $birthdate\n";
    $mensaje .= "Dirección: $address\n";
    $mensaje .= "Región: $region\n";
    $mensaje .= "Curso de Verano Precalculo $subscription grupo de dudas\n";
    $mensaje .= "Código de descuento utilizado: $discount_code\n";

    $headers  = "From: " . $nombre_remitente . " <" . $correo_remitente . ">\r\n";
    $headers .= "Reply-To: " . $correo_remitente . "\r\n";
    $headers .= "Content-Type: text/plain; charset=utf-8\r\n";
    $headers .= "MIME-Version: 1.0\r\n";

    return @mail($destinatario, $asunto, $mensaje, $headers);
}

// --- MAIN ---
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    json_exit(['error' => 'Método no permitido.'], 405);
}

// Lee POST seguro (sin warnings si falta algún campo)
$fields = [
  'name','lastname','rut','phone','email','birthdate','address','region','subscription','has_discount','discount_code'
];
$post = [];
foreach ($fields as $f) {
  $post[$f] = isset($_POST[$f]) ? trim((string)$_POST[$f]) : '';
}

// Sanitiza
$name         = htmlspecialchars($post['name']);
$lastname     = htmlspecialchars($post['lastname']);
$rut          = htmlspecialchars($post['rut']);
$phone        = htmlspecialchars($post['phone']);
$email        = htmlspecialchars($post['email']);
$birthdate    = htmlspecialchars($post['birthdate']);
$address      = htmlspecialchars($post['address']);
$region       = htmlspecialchars($post['region']);
$subscription = htmlspecialchars($post['subscription']);
$has_discount = htmlspecialchars($post['has_discount']);

// Requeridos
if ($name==='' || $lastname==='' || $rut==='' || $phone==='' || $email==='' || $birthdate==='' || $address==='' || $region==='' || $subscription==='' || $has_discount==='') {
    json_exit(['error' => 'Por favor, completa todos los campos requeridos.'], 400);
}

// Código descuento
$discount_code = '';
if ($has_discount === 'si') {
    $discount_code = htmlspecialchars($post['discount_code']);
    if ($discount_code === '' || !verificarCodigoDescuento($discount_code)) {
        json_exit(['error' => 'El código de descuento ingresado no es válido o está vacío.'], 400);
    }
}

// Datos para CSV
$fechaRegistro = date('Y-m-d H:i:s');
$datosCSV = [$fechaRegistro, $name, $lastname, $rut, $phone, $email, $birthdate, $address, $region, $subscription, $discount_code];

// Escribe archivos (si falla, te lo dice en JSON, sin romper nada)
if (!escribirCSV($datosCSV)) {
    json_exit(['error' => 'No se pudo escribir registros.csv (revisa permisos/propietario de la carpeta).'], 500);
}

$txt  = "Registro realizado el: $fechaRegistro\n";
$txt .= "Nombre(s): $name\n";
$txt .= "Apellidos: $lastname\n";
$txt .= "RUT: $rut\n";
$txt .= "Teléfono: +56 9 $phone\n";
$txt .= "Correo Electrónico: $email\n";
$txt .= "Fecha de nacimiento: $birthdate\n";
$txt .= "Dirección: $address\n";
$txt .= "Región: $region\n";
$txt .= "Curso de Verano Precalculo $subscription grupo de dudas\n";
$txt .= "Código de descuento utilizado: $discount_code\n";
$txt .= "--------------------------\n";

if (!escribirTXT($txt)) {
    json_exit(['error' => 'No se pudo escribir registros.txt.'], 500);
}

// Correo (opcional: si falla, igual puedes seguir a pago)
if (!enviarCorreo($name, $lastname, $rut, $phone, $email, $birthdate, $address, $region, $subscription, $discount_code)) {
    // Si quieres que NO bloquee el pago cuando falle mail, deja esto así:
    // json_exit(['error' => 'Hubo un error al enviar el correo.'], 500);
}

// Redirect URL
$redirect = ($subscription === 'sin')
    ? ($has_discount === 'si' ? 'https://janopaes.cl/NoTeEchesCalculo/Sin/Cod/' : 'https://janopaes.cl/NoTeEchesCalculo/Sin/')
    : ($has_discount === 'si' ? 'https://janopaes.cl/NoTeEchesCalculo/Con/Cod/' : 'https://janopaes.cl/NoTeEchesCalculo/Con/');

json_exit(['redirect_url' => $redirect]);
